#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "touch.h"

#ifndef UNITTEST
#include "input/mtouch_log.h"
#include "input/mtouch_driver.h"
#include "input/event_types.h"
#else
#include "timer_header.h"
#endif

#define _NUM_CHARS_PER_BYTE 5    /* Each byte takes 5 chars to display in slog */


/* TODO ERICK: Group the command functions (operating mode and config/test mode into this common one */
int
cypress_send_command(cypress_dev_t* cypress)
{
    return EOK;
}

int
cypress_check_error_pin(cypress_dev_t* cypress)
{
    uint8_t *reg;
    int fatality = 0;
    int i;

    reg =(uint8_t *) calloc(2, sizeof(uint8_t));
    /* Check register */
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, ERROR_STATUS_LOW, 2, reg)) < 0) {
        mtouch_error (cypress->log_name, "Failed to read error status register");
        return -1;
    }

    cypress->error_pin_status |= *reg;

    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, ERROR_STATUS_HIGH, 2, reg)) < 0) {
        mtouch_error (cypress->log_name, "Failed to read error status register");
        return -1;
    }

    cypress->error_pin_status |= *reg << 8;

    for (i = 0; i < 8; i++) {
        switch (cypress->error_pin_status & (1 << i)) {
        case ERROR_WATCH_DOG_TIMER:
            break;
        case ERROR_VDDA_LOW_VOLATAGE:
            break;
        case ERROR_CONFIG_CRC:
            mtouch_critical (cypress->log_name, "Configuration data in flash is corrupted.  Halting driver");
            error_memory("Cypress_Touch: Configuration data in flash is corrupted.  Halting driver");
            fatality = 1;
            break;
        case ERROR_BOOTLOADER_CRC:
            mtouch_critical (cypress->log_name, "Bootloader in flash is corrupted.  Halting driver");
            error_memory("Cypress_Touch: Bootloader in flash is corrupted.  Halting driver");
            fatality = 1;
            break;
        case ERROR_APPLICATION_CRC:
            mtouch_critical (cypress->log_name, "Application in flash is corrupted.  Driver waiting for new firmware...");
            error_memory("Cypress_Touch: Application in flash is corrupted.  Driver waiting for new firmware...");
            /* Reset the controller */
            mtouch_error (cypress->log_name, "Check error pin App CRC error, resetting ...");
            cypress_controller_reset(cypress);  // IS THIS NEEEDED?
            cypress_bootloader_clear_error_pin(cypress);
            cypress->bootloader_state =  CLEARING_ERROR;
            return ERROR_RECOVERABLE;
        case ERROR_INVALID_CONFIG:
            mtouch_critical (cypress->log_name, "Configuration data in flash does not fit device capabilities.  Halting driver");
            error_memory ("Cypress_Touch: Configuration data in flash does not fit device capabilities.  Halting driver");
            fatality = 1;
            break;
        case ERROR_AUTO_SHORT_TEST:
            mtouch_critical (cypress->log_name, "Short Circuit detected.  FPC, panel, or PCB hardware error.  Halting driver");
            error_memory ("Cypress_Touch: Short Circuit detected.  FPC, panel, or PCB hardware error.  Halting driver");
            /* Raiden wins! */
            fatality = 1;
            break;
        case ERROR_OPEN_CIRCUIT_TEST:
            mtouch_error (cypress->log_name, "Open Circuit detected. FPC, panel, or PCB hardware error.");
            /* Nothing to do, it is a persistant error but not fatal */
            break;
        default:
            /* Reserved */
            break;
        }
    }

    /* Doesn't matter if we can there are recoverable errors if halt condition occured */
    if (fatality) {
        return ERROR_NON_RECOVERABLE;
    }

    /* Recoverable errors are only within the bootloader */
    if (BOOTLOADER_MODE == cypress_get_mode(cypress))
        cypress_bootloader_clear_error_pin(cypress);
    else
        cypress_config_clear_error_pin(cypress);

    /* Return EOK regarless if clearing the error worked or not (except non recoverable errors).  If the error pin is not cleared we will just get
    back into this handler again. Depending on if there is an issue with the hardware this may result in an infinte loop. */
    free(reg);
    return ERROR_RECOVERABLE;
}

int
cypress_display_packet(cypress_dev_t* cypress, char *direction, uint8_t reg, uint8_t *packet, int len)
{
    int i = 0;
    char *buf;
    int remaining = len;
    int offset = 0;
    int chars_to_display = 0;

    /* Check to see if it exceeds the 80 char limit, if so break up the packet */
    if ((len * _NUM_CHARS_PER_BYTE) > cypress->slog_char_limit) {
        /* Split it up into 80 or less char strings */
        do {
            chars_to_display = cypress->slog_char_limit / _NUM_CHARS_PER_BYTE;
            if (-1 == cypress_display_packet(cypress, direction, (reg + offset), &packet[offset], chars_to_display))
                return -1;

            offset += chars_to_display;
            remaining -= chars_to_display;
        } while ((remaining * _NUM_CHARS_PER_BYTE) > cypress->slog_char_limit);

        /* Finish up the rest... */
        len = remaining;
    }

    /* 5 is the length of the string '0x%x ' below... */
    buf = (char *) calloc ((_NUM_CHARS_PER_BYTE * len), sizeof(char));

    if (buf == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate buffer space to dump touch record");
        return -1;
    }

    sprintf (buf, "0x%x ", packet[offset + i]);
    i++;
    for (; i < len; i++) {
        sprintf ((buf + strlen(buf)), "0x%x ", packet[offset + i]);
    }

    mtouch_info (cypress->log_name, "%s: reg: 0x%x: %s", direction, (reg + offset), buf);

    free (buf);

    return EOK;
}

int
cypress_get_mode(cypress_dev_t* cypress)
{
    uint8_t reg[2] = {0};
    int mode = -1;

    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, 2, reg)) == 0) {
        if (reg[RESET_DETECT] != 0) {
            mode = BOOTLOADER_MODE;

            if (cypress->verbose > 6)
                mtouch_info (cypress->log_name, "Current mode is: Bootloader");
        } else if (((reg[HST_MODE] & DEVICE_MODE_MASK) == SYSTEM_INFORMATION_MODE) && (reg[RESET_DETECT] == 0)) {
            mode = SYSTEM_INFORMATION_MODE;

            if (cypress->verbose > 6)
                mtouch_info (cypress->log_name, "Current mode is: System Information");
        } else if (((reg[HST_MODE] & DEVICE_MODE_MASK) == OPERATING_MODE) && (reg[RESET_DETECT] == 0)) {
            mode = OPERATING_MODE;

            if (cypress->verbose > 6)
                mtouch_info (cypress->log_name, "Current mode is: Operating");
        } else if (((reg[HST_MODE] & DEVICE_MODE_MASK) == CONFIGURATION_AND_TEST_MODE) && (reg[RESET_DETECT] == 0)) {
            mode = CONFIGURATION_AND_TEST_MODE;

            if (cypress->verbose > 6)
                mtouch_info (cypress->log_name, "Current mode is: Configure and Test");
        } else {
            /* Unknown mode! */
            mtouch_error (cypress->log_name, "Controller is in an unknown mode: [%x]!  Driver State [%d]", reg[HST_MODE], cypress->driver_state);
            error_memory ("Cypress_Touch: Controller is in an unknown mode: [%x]!  Driver State [%d]", reg[HST_MODE], cypress->driver_state);
            mode = UNKNOWN;
        }
    } else {
        mtouch_error (cypress->log_name, "Failed to read touch controller HST_MODE");
        error_memory ("Cypress_Touch: Failed to read touch controller HST_MODE");
        mode = -1;
    }

    return (mode);
}

int
cypress_set_mode(cypress_dev_t* cypress, uint8_t mode)
{
    int rc = 0;
    uint8_t reg = 0;
#ifndef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    int count = 0;
#endif
    uint8_t buf = 0;

    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, sizeof(uint8_t), &reg)) < 0) {
        mtouch_error (cypress->log_name, "Failed to read touch controller HST_MODE");
        return -1;
    }

    buf = (mode | (reg & MODE_MASK));
    buf |= MODE_CHANGE_REQUEST;

    rc = cypress->i2c_funcs.write_reg(cypress->i2c_fd, HST_MODE, sizeof(buf), &buf);
    if (EOK != rc) {
        mtouch_error(cypress->log_name, "Failed to set HST MODE: %x", buf);
        return -1;
    }

#ifndef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    if(mode != CONFIGURATION_AND_TEST_MODE) {
        /* Wait until we have switched modes */
        while (mode != cypress_get_mode(cypress)) {
               delay (10);
               count++;

               if (count > 100)
                   return -1;
        }
    }
#endif
    /* The controller appears to be ready based on the HST_MODE register, however it's been noticed that often commands
       sent fail in odd ways.  The controller reports them completed, however data yielded is often garbage.  Placing an
       additional delay prior to issuing the command helps alleviate this. */
    delay (1);

    return EOK;
}

int
cypress_reply_ready(cypress_dev_t* cypress)
{
    uint8_t data = 0;
    int i;

    for (i = 0; i < cypress->command_reply_retry_limit; i++) {
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, COMMON_COMMAND_OFFSET, sizeof(uint8_t), &data)) < 0) {
            mtouch_error (cypress->log_name, "Failed to read touch controller status");
            goto fail;
        }

        if (data & COMMAND_COMPLETE) {
            if (cypress->verbose > 5)
                mtouch_info (cypress->log_name, "Controller has processed command, response is waiting");

            return EOK;
        }

        delay (1000);
    }

    mtouch_warn (cypress->log_name, "Controller took to long too reply to command");

fail:

    return -1;
}

int
cypress_read_command_reply(cypress_dev_t* cypress, uint8_t **reply, uint16_t len)
{
    uint8_t *buf;
    int data_offset = 0;
    int bytes_to_read = 0;
    int bytes_remaining = len;

    if (*reply != NULL) {
        free (*reply);
        *reply = NULL;
    }

    *reply =(uint8_t *) calloc (len, sizeof (uint8_t));

    if (*reply == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for command data response");
        return -1;
    }

    do {
        bytes_to_read = 0;
        /* Reply might be larger than the max read */
        if ((bytes_remaining + COMMON_COMMAND_DATA_OFFSET) > CYPRESS_MAX_REGLEN) {
            bytes_to_read = (CYPRESS_MAX_REGLEN - COMMON_COMMAND_DATA_OFFSET);
        } else {
            bytes_to_read += bytes_remaining;
        }

        /* Read the reply from the command */
        buf =(uint8_t *) calloc(bytes_to_read + COMMON_COMMAND_DATA_OFFSET, sizeof(uint8_t));
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, (bytes_to_read + COMMON_COMMAND_DATA_OFFSET), buf)) < 0) {
            mtouch_error (cypress->log_name, "Failed to read Retrieve Data output");
            return -1;
        }

        memcpy ((*reply + data_offset), (buf + COMMON_COMMAND_DATA_OFFSET), bytes_to_read);
        bytes_remaining -= bytes_to_read;
        data_offset += bytes_to_read;

        if (cypress->verbose > 5)
            cypress_display_packet(cypress, "r", HST_MODE, buf, (bytes_to_read + COMMON_COMMAND_DATA_OFFSET));

    } while (bytes_remaining > 0);

    free(buf);
    return len;
}


int
cypress_soft_reset(cypress_dev_t* cypress)
{
    int rc = 0;
    uint8_t buf = 0;
    struct itimerspec       itime;

    buf = buf | RESET;

    rc = cypress->i2c_funcs.write_reg(cypress->i2c_fd, HST_MODE, sizeof(buf), &buf);
    if (0 != rc) {
        mtouch_error(cypress->log_name, "Soft reset failed: Failed to set HST MODE: %x retval %d", buf, rc);
        return -1;
    }

    /* If Polling mode is enabled, disable it for now */
    if (cypress->polling_rate) {
        timer_create (CLOCK_REALTIME, &cypress->tp_intrevent, &cypress->timer_id);

        itime.it_value.tv_sec = 0;
        itime.it_value.tv_nsec = 0;
        itime.it_interval.tv_sec = 0;
        itime.it_interval.tv_nsec = 0;
        timer_settime(cypress->timer_id, 0, &itime, NULL);
    }

    /* Delay waiting for the controller */
    delay (cypress->soft_reset_recovery_delay);

    if (cypress->polling_rate) {
        /* Re-enable polling mode */
        timer_create (CLOCK_REALTIME, &cypress->tp_intrevent, &cypress->timer_id);

        itime.it_value.tv_sec = 0;
        itime.it_value.tv_nsec = cypress->polling_rate * MSEC;
        itime.it_interval.tv_sec = 0;
        itime.it_interval.tv_nsec = cypress->polling_rate * MSEC;
        timer_settime(cypress->timer_id, 0, &itime, NULL);
    }

    return EOK;
}

void
cypress_update_pps_status(cypress_dev_t* cypress, char *status)
{
    if (-1 != cypress->pps_status_fd){
        if (-1 == write(cypress->pps_status_fd, status, strlen(status)))
            mtouch_warn (cypress->log_name, "Write to PPS status object '%s' failed: %d", status, errno);
	}
}

/* TODO ERICK */
int
cypress_reset_detect()
{
    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/common.c $ $Rev: 886397 $")
#endif
